/******************************************************************************
 * (C) Copyright 2000 by Agilent Technologies GmbH. All rights reserved.      *
 ******************************************************************************/

/* ---------------------------------------------------------------
 * File: xrt.h 
 *       Requester-target (RT) functions
 * -----------------------------------------------------------------*/

#include <xpciapi.h>

#include <xregcons.h>
#include <xregxdir.h>
#include <xregx10.h>
#include <xregx11.h>

#if 0 /* HL, not needed??? */
#  include <xpci.h>
#  include <timeout.h>
#endif

#include <xaddrmap.h>

/* Pointer to RT database */
#define DB_RT (bx_handlearray[handle].db->Exerciser.RequesterTarget)

/* Value of generic property */
#define GENERIC(prop) (DB_RT.GenProp[prop])

/* Pointer to one line in behavior-memory */
#define BEHAVIOR(line) (DB_RT.Behavior.Mem[line])

/* Pointer to Behavior Update Info */
#define BEH_UPD (&DB_RT.Behavior.UpdateInfo)

/* memory map moved to xdyndata.c */

/* the following lines leaves running code running.... */
extern bx_memmaptype Mephisto_RT_Beh2Hw [];
#define Beh2Hw Mephisto_RT_Beh2Hw

/*****************************************************************************
 * RT Programming
 ****************************************************************************/

/*---------------------------------------------------------------------------*
 * BestXRTProg
 *
 * Purpose: Program properties and memories from host to card
 *---------------------------------------------------------------------------*/
bx_errtype BestXRTProg (bx_handletype handle)
{
  BX_DECLARE_FUNCNAME("BestXRTProg [rtprog]");

  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    /* Switch to Prog-Mode (DBI clock) */
    BX_TRY_PROGRESS(BestXExerciserProgMode(handle,0 /* no soft force */));

    BX_TRY(BestXRTBehProg(handle));
    BX_TRY(BestXRTGenProg(handle));

    /* Switch to Run-Mode (PCI clock) */
    BX_TRY_PROGRESS(BX_E_OK);
    BX_TRY_PROGRESS(BestXExerciserRunMode(handle,0 /* no soft force */));
  }

  BX_TRY_CATCH
  {
    BX_TRY_PASSED
    {
      /* Switch to progmode successful */
      BX_TRY_PASSED
      {
        /* all functions in between successful */
        BX_TRY_PASSED
        {
          /* switch to runmode successful */
        }
        BX_TRY_FAILED
        {
          /* switch to runmode not successful */
          (void) BestXExerciserRunMode(handle, 1 /* soft force */);
        }
      }
      BX_TRY_FAILED
      {
        /* some function in between not successful */
        (void) BestXExerciserRunMode(handle, 0 /* no soft force */);
      }
    }
    BX_TRY_FAILED
    {
      /* switch to progmode not successful */
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXRTRead
 *
 * Purpose: Read properties and memories from card- to host memory 
 *---------------------------------------------------------------------------*/
bx_errtype BestXRTRead (bx_handletype handle,bx_int32 option)
{
  BX_DECLARE_FUNCNAME("BestXRTRead [rtread]");

  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    BX_TRY(BestXRTBehRead(handle,option));
    BX_TRY(BestXRTGenRead(handle));
  }

  BX_ERRETURN(BX_TRY_RET);
}


/*****************************************************************************
 * RT generics
 ****************************************************************************/
 
/*---------------------------------------------------------------------------*
 * BestXRTGenGet 
 *
 * Purpose: Gets a generic property from host-memory 
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXRTGenGet(
  bx_handletype handle,
  bx_rtgentype prop,
  bx_int32 * val
)
{
  BX_DECLARE_FUNCNAME("BestXRTGenGet [rtgenget]");

  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    /* range checking */
    BX_TRY(BestXParamProbe(handle, BX_PARAM_RT_GEN, (bx_int32)prop));
    BX_TRY_FCT_PARAM_NULL_POINTER(val);
     
    switch (prop)
    {
      case BX_RTGEN_NUMBEH:

        /* BX_RTGEN_NUMBEH does not exist in HW.
           If user gets this property, he
           implicitly gets it from Group 0
           (all groups usually have same value):
        */
        BX_TRY(BestXRTGenGet(handle, BX_RTGEN_NUMBEHG0, val));
        break;

      default:
        /* get property from host DB */
        *val=(bx_int32)GENERIC(prop);
        break;
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXRTGenSet         
 *
 * Purpose: Sets a generic property (on host only) 
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXRTGenSet(
  bx_handletype handle,
  bx_rtgentype prop,
  bx_int32 val)

{
  BX_DECLARE_FUNCNAME("BestXRTGenSet [rtgenset]");

  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    /* range checking */
    BX_TRY(BestXParamCheck(handle, BX_PARAM_RT_GEN, (bx_int32)prop, val));
     
    switch (prop)
    {
      case BX_RTGEN_NUMBEH:

        /* BX_RTGEN_NUMBEH does not exist in HW.
           If user sets this property, he
           implicitly sets all groups:
        */
        BX_TRY(BestXRTGenSet(handle, BX_RTGEN_NUMBEHG0, val));
        BX_TRY(BestXRTGenSet(handle, BX_RTGEN_NUMBEHG1, val));
        BX_TRY(BestXRTGenSet(handle, BX_RTGEN_NUMBEHG2, val));
        BX_TRY(BestXRTGenSet(handle, BX_RTGEN_NUMBEHG3, val));
        BX_TRY(BestXRTGenSet(handle, BX_RTGEN_NUMBEHG4, val));
        break;

      default:
        /* set property in host DB */
        GENERIC(prop)=val;
        break;
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXRTGenDefaultSet
 *
 * Purpose: Set all generic properties to default (on host only) 
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXRTGenDefaultSet(
  bx_handletype handle)

{
  BX_DECLARE_FUNCNAME("BestXRTGenDefaultSet [rtgendefset]");

  BX_TRY_VARS;
  bx_int32 i;

  /* Generic info for range checking */
  const bx_generic_infotype *GenInfo;
  const bx_param_infotype *ParamInfo;

  BX_TRY_BEGIN
  {
    /* Get pointer to generic info */
    BX_TRY(BestXGenInfoGet(handle, BX_PARAM_RT_GEN, &GenInfo));

    for (i = 0; i < GenInfo->num_elem; i++)
    {
      /* Get pointer to i-th (existing) property */
      BX_TRY(BestXParamInfoGet(handle, BX_PARAM_RT_GEN,
            i, &ParamInfo, (bx_int32)BX_INDEX_SEARCH));

      /* Set it to default */
      BX_TRY(BestXRTGenSet(handle,
              ParamInfo->proptyp.rtgenprop,
              ParamInfo->defaultval));
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXRTGenProg
 *
 * Purpose: Writes all generic properties to card
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXRTGenProg(bx_handletype  handle)
{
  /* Updates Generics memory on card */

  BX_DECLARE_FUNCNAME("BestXRTGenProg [rtgenprog]");

  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    bx_int32 value=0;

    /* Switch to Prog-Mode (DBI clock) */
    BX_TRY_PROGRESS(BestXExerciserProgMode(handle,0 /* no soft force */));

    BX_TRY(BestXRTGenGet (handle,  BX_RTGEN_NUMBEHG0, &value));
    BX_TRY(BestXDirectRegWrite(handle, BX_REG_RTBEHAV_CTR0_LOOP_REG,2, value-1));

    BX_TRY(BestXRTGenGet (handle,  BX_RTGEN_NUMBEHG1, &value));
    BX_TRY(BestXDirectRegWrite(handle, BX_REG_RTBEHAV_CTR1_LOOP_REG,2, value-1));

    BX_TRY(BestXRTGenGet (handle,  BX_RTGEN_NUMBEHG2, &value));
    BX_TRY(BestXDirectRegWrite(handle, BX_REG_RTBEHAV_CTR2_LOOP_REG,2, value-1));

    BX_TRY(BestXRTGenGet (handle,  BX_RTGEN_NUMBEHG3, &value));
    BX_TRY(BestXDirectRegWrite(handle, BX_REG_RTBEHAV_CTR3_LOOP_REG,2, value-1));

    BX_TRY(BestXRTGenGet (handle,  BX_RTGEN_NUMBEHG4, &value));
    BX_TRY(BestXDirectRegWrite(handle, BX_REG_RTBEHAV_CTR4_LOOP_REG,2, value-1));

    /* Switch to Run-Mode (PCI clock) */
    BX_TRY_PROGRESS(BX_E_OK);
    BX_TRY_PROGRESS(BestXExerciserRunMode(handle,0 /* no soft force */));
  }
  
  BX_TRY_CATCH
  {
    BX_TRY_PASSED
    {
      /* Switch to progmode successful */
      BX_TRY_PASSED
      {
        /* all functions in between successful */
        BX_TRY_PASSED
        {
          /* switch to runmode successful */
        }
        BX_TRY_FAILED
        {
          /* switch to runmode not successful */
          (void) BestXExerciserRunMode(handle, 1 /* soft force */);
        }
      }
      BX_TRY_FAILED
      {
        /* some function in between not successful */
        (void) BestXExerciserRunMode(handle, 0 /* no soft force */);
      }
    }
    BX_TRY_FAILED
    {
      /* switch to progmode not successful */
    }
  }


  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXRTGenRead
 *
 * Purpose: Reads all generic properties from card- to host-memory
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXRTGenRead(bx_handletype  handle)
{
  /* Updates generics memory on host DB */

  BX_DECLARE_FUNCNAME("BestXRTGenRead [rtgenread]");

  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    bx_int32 value=0;

    /* Switch to Prog-Mode (DBI clock) */
    BX_TRY_PROGRESS(BestXExerciserProgMode(handle,0 /* no soft force */));

    BX_TRY(BestXDirectRegRead(handle, BX_REG_RTBEHAV_CTR0_LOOP_REG, sizeof(bx_int16), &value));
    BX_TRY(BestXRTGenSet (handle,  BX_RTGEN_NUMBEHG0, value+1));
    
    /* Arbitrary: take value from group 0 */
    BX_TRY(BestXRTGenSet (handle,  BX_RTGEN_NUMBEH, value+1));
    
    BX_TRY(BestXDirectRegRead(handle, BX_REG_RTBEHAV_CTR1_LOOP_REG, sizeof(bx_int16), &value));
    BX_TRY(BestXRTGenSet (handle,  BX_RTGEN_NUMBEHG1, value+1));
    
    BX_TRY(BestXDirectRegRead(handle, BX_REG_RTBEHAV_CTR2_LOOP_REG, sizeof(bx_int16), &value));
    BX_TRY(BestXRTGenSet (handle,  BX_RTGEN_NUMBEHG2, value+1));
    
    BX_TRY(BestXDirectRegRead(handle, BX_REG_RTBEHAV_CTR3_LOOP_REG, sizeof(bx_int16), &value));
    BX_TRY(BestXRTGenSet (handle,  BX_RTGEN_NUMBEHG3, value+1));

    BX_TRY(BestXDirectRegRead(handle, BX_REG_RTBEHAV_CTR4_LOOP_REG, sizeof(bx_int16), &value));
    BX_TRY(BestXRTGenSet (handle,  BX_RTGEN_NUMBEHG4, value+1));

    /* Switch to Run-Mode (PCI clock) */
    BX_TRY_PROGRESS(BX_E_OK);
    BX_TRY_PROGRESS(BestXExerciserRunMode(handle,0 /* no soft force */));
  }
  
  BX_TRY_CATCH
  {
    BX_TRY_PASSED
    {
      /* Switch to progmode successful */
      BX_TRY_PASSED
      {
        /* all functions in between successful */
        BX_TRY_PASSED
        {
          /* switch to runmode successful */
        }
        BX_TRY_FAILED
        {
          /* switch to runmode not successful */
          (void) BestXExerciserRunMode(handle, 1 /* soft force */);
        }
      }
      BX_TRY_FAILED
      {
        /* some function in between not successful */
        (void) BestXExerciserRunMode(handle, 0 /* no soft force */);
      }
    }
    BX_TRY_FAILED
    {
      /* switch to progmode not successful */
    }
  }



  BX_ERRETURN(BX_TRY_RET);
}


/*****************************************************************************
 * RT Behavior
 ****************************************************************************/

/*---------------------------------------------------------------------------*
 * BestXRTBehMemInit
 *
 * Purpose: Defaults the complete behavior memory (on host only)
 * Caution: A following ExerciserProg() will re-program whole memory !!
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXRTBehMemInit(
    bx_handletype handle)
{
  BX_DECLARE_FUNCNAME ("BestXRTBehMemInit [rtbehmeminit]");

  BX_TRY_VARS;
  bx_int32 offset;
  
  BX_TRY_BEGIN
  {
    for (offset=0;offset<BX_RTBEH_MEMDEPTH;offset++)
    {
      BX_TRY(BestXRTBehDefaultSet(handle,offset));
    }
  }
  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXRTBehDefaultSet
 *
 * Purpose: Defaults one behavior line (on host)
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXRTBehDefaultSet(
  bx_handletype handle,
  bx_int32 offset)
{
  BX_DECLARE_FUNCNAME("BestXRTBehDefaultSet [rtbehdefset]");

  BX_TRY_VARS;
  bx_int32 i;

  /* Generic info for range checking */
  const bx_generic_infotype *GenInfo;
  const bx_param_infotype *ParamInfo;

  BX_TRY_BEGIN
  {
    /* Get pointer to generic info */
    BX_TRY(BestXGenInfoGet(handle, BX_PARAM_RT_BEH, &GenInfo));

    for (i = 0; i < GenInfo->num_elem; i++)
    {
      /* Get pointer to i-th (existing) property */
      BX_TRY(BestXParamInfoGet(handle, BX_PARAM_RT_BEH,
            i, &ParamInfo, (bx_int32)BX_INDEX_SEARCH));

      /* Set it to default */
      BX_TRY(BestXRTBehSet(handle,offset,
              ParamInfo->proptyp.rtbehprop,
              ParamInfo->defaultval));
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXRTBehGet
 *
 * Purpose: Gets a behavior property (from host memory)
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXRTBehGet(
  bx_handletype handle,
  bx_int32 offset,
  bx_rtbehtype prop,
  bx_int32 * val)
{
  BX_DECLARE_FUNCNAME("BestXRTBehGet [rtbehget]");

  bx_int32 i;
  bx_int32 ival;  /* partial bits of property value */

  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    /* Get pointer to i-th (existing) property */
    BX_TRY(BestXParamProbe(handle, BX_PARAM_RT_BEH, (bx_int32)prop));
    BX_TRY_FCT_PARAM_NULL_POINTER(val);

    /* Walk through all entries in table Beh2Hw[].
       There is at least one entry for each property. 
    */
    
    *val=0;
    i=0;
    do   /* Table Beh2Hw[] has at least one valid entry */
    {
      if (Beh2Hw[i].PropId==(bx_int32)prop)
      {
        /* This is an entry for property prop; we get corresponding bits */       
        ival=0;
        if (!BestXMemPropGet
        (
          (bx_int32*)(BEHAVIOR(offset)+Beh2Hw[i].MemBitPos/16), /* pointer memory location to change */
          Beh2Hw[i].MemBitPos%16,                     /* first bitpos to read */           
          Beh2Hw[i].MemBitPos%16+Beh2Hw[i].PropLen-1, /* last bitpos to read */
          &ival,                                      /* returned bits */
          Beh2Hw[i].PropName,                         /* property name for debugging */
          sizeof(bx_int16)
        ))
        {
          assert(0); 
          BX_ERRETURN(BX_E_INVALID_CASE); /* fatal parameter error, should never happen */
        }

        /* merge read bits of ival into *val */
        *val = *val | ival<<Beh2Hw[i].PropBitPos; /* set bits */
      }
    } while (Beh2Hw[++i].PropId!=MEMMAP_LASTENTRY);
  }

  /* We need some modifications to adapt sw to hw */
  switch (prop)    
  {
    case BX_RTBEH_REPEAT:
      if (*val==0)
      {
        /* This register is 16 bit wide in HW.
           A value of zero in HW means 65536 in CAPI.
        */
        *val=0x10000UL;
      }
      break;
    
    case BX_RTBEH_LATENCY:
        /* User gets at least a value of three here.
           A three here in CAPI is represented with a zero
           in HW. So we have to increment by three.
        */ 
        *val += 3;         /* hw automatically adds 3 */
      break;

    default:
      break;
  }

  BX_ERRETURN(BX_E_OK);
}

/*---------------------------------------------------------------------------*
 * BestXRTBehSet
 *
 * Purpose: Sets a behavior property (host memory)
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXRTBehSet(
  bx_handletype handle,
  bx_int32 offset,
  bx_rtbehtype prop,
  bx_int32 val
)

{
  BX_DECLARE_FUNCNAME("BestXRTBehSet [rtbehset]");
  bx_int32 i;
 
  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    /* dynamic capability checking concerning values of parameters */
    BX_TRY(BestXParamCheck(handle,
               BX_PARAM_RT_BEH, (bx_int32)prop, val));

     /* We need some modifications to adapt sw to hw */
    switch (prop)        
    {
      case BX_RTBEH_REPEAT:
        if (val==0x10000UL)
        {
          /* This register is 16 bit wide in HW.
             A value of zero in HW means 65536 in CAPI
          */
          val=0UL;
        }
        break;
      
      case BX_RTBEH_LATENCY:
        /* User programs at least a value of three here.
           A three here in CAPI is represented with a zero
           in HW. So we have to decrement by three.
        */ 
        assert (val >= 3);   /* granted by dyncaps */  
        val -= 3;         /* hw automatically adds 3 */
        break;
      
      default:
        break;
    }

    /* Walk through all entries in table Beh2Hw[].
       There is at least one entry for each property. 
    */

    i=0;
    do   /* Table Beh2Hw[] has at least one valid entry */
    {
      if (Beh2Hw[i].PropId==(bx_int32)prop)
      {
        /* This is an entry for property prop; we set corresponding bits */       
 
        if (BestXMemPropSet
        (
          (bx_int32*)(BEHAVIOR(offset)+Beh2Hw[i].MemBitPos/16), /* pointer memory location to change */
          Beh2Hw[i].MemBitPos%16,               /* first bitpos to change */           
          Beh2Hw[i].MemBitPos%16+Beh2Hw[i].PropLen-1, /* last bitpos to change */
          val>>Beh2Hw[i].PropBitPos, /* relevant bits to set start at position 0 */
          Beh2Hw[i].PropName,        /* property name for debugging */
          sizeof(bx_int16)
        ))
        {
          /* success */
          /* mark this property modified */
          SetUpdateInfo(BEH_UPD,1,offset,Beh2Hw[i].MemBitPos/16);  
        }
        else
        {
          assert(0); 
          BX_ERRETURN(BX_E_INVALID_CASE); /* fatal parameter error, should never happen */
        }
      }
    } while (Beh2Hw[++i].PropId!=MEMMAP_LASTENTRY);
  }
  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXRTBehProg
 *
 * Purpose: Writes changes of behavior memory from host to card
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXRTBehProg(bx_handletype  handle)
{
  /* Updates RT behavior on card */
  bx_int32 decspeed=0;
  bx_int32 busspeed=0;

  BX_DECLARE_FUNCNAME("BestXRTBehProg [rtbehprog]");

  BX_TRY_VARS;

  bx_bool  Touched;  /* Wether memory needs an update at all */
  bx_int32 FirstRow,LastRow,FirstCol,LastCol; /* Range to update */

  bx_int32 i,row,col;  /* Indices */

  bx_int16ptr data=NULL;    /* part of behavior memory to be written to card */
  bx_int32 NumWords;   /* size of data[] in words */

  /* Retrieve info about the range to update */
  GetUpdateInfo(BEH_UPD,&Touched,&FirstRow,&LastRow,&FirstCol,&LastCol);
  
  BX_TRY_BEGIN
  {
    if (!Touched)
    {
      /* host and card memory are identical, no update needed */
    }
    else
    {
      /* Host- and card memory differ */

      /* Decode speed A does not work in E2930B at all, i.e. bit 0 of BX_REG_GEN_DEC_PROP_REG 
         must never be set (else ACK64# may erronously be asserted).
       */
      if (bx_handlearray[handle].hwinfo.hw!=BX_HW_E2930B)
      {
        /* Decode speed A works only, if busfrequency is below 67 MHz */
        if (FirstCol <= 4  && 4 <= LastCol) /* decspeed is in memory 4 (see xdyndata.c) */
        {
          /* Decoderspeed will be programmed in some row */
          for (row=FirstRow;row<=LastRow;row++)
          {
            BX_TRY(BestXRTBehGet(handle,row,BX_RTBEH_DECSPEED,&decspeed));
            if (decspeed==BX_RTBEH_DECSPEED_A)
            {
              /* check current frequency */
              BX_TRY(BestXStatusRead(handle,BX_STAT_BUSSPEED,&busspeed));
              if(busspeed >= BX_MAXBUSSPEED_FOR_DECSPEED_A)
              {
                /* Frequency is too high, enable automatic HW switching to decode speed B (HW default) */
                BX_TRY(BestXDirectRegMaskWrite(handle,BX_REG_GEN_DEC_PROP_REG,sizeof(bx_int16),1,0));
              }
              else
              {
                /* Frequency is low enough; enable decode speed A in HW */
                BX_TRY(BestXDirectRegMaskWrite(handle,BX_REG_GEN_DEC_PROP_REG,sizeof(bx_int16),1,1));
              }
              break;  /* Skip following rows */
            }
          }
        }
      }

      /* Switch to Prog-Mode (DBI clock) */
      BX_TRY_PROGRESS(BestXExerciserProgMode(handle,0 /* no soft force */));
    
      /* Min and Max registers determine the Column (property) -range 
         to program 
      */
      assert(FirstCol<BX_RTBEH_MEMWIDTH);
      assert(LastCol<BX_RTBEH_MEMWIDTH);
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_RTBEHAV_MIN_REG,sizeof(bx_int16),FirstCol));
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_RTBEHAV_MAX_REG,sizeof(bx_int16),LastCol));

      /* Set starting row (i.e. offset) from which to start programming */
      assert(FirstRow<BX_RTBEH_MEMDEPTH);
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_RTBEHAV_CTR,sizeof(bx_int16),FirstRow));

      /* In order to use a single call to BestXDirectRegBlockWrite() 
         instead of many calls to BestXDirectRegWrite(),
         we first have to copy the needed behavior properties into 
         a local data array 
      */
      
      NumWords=(LastRow-FirstRow+1) * (LastCol-FirstCol+1);
      data=(bx_int16ptr) BestXMemCalloc(NumWords,sizeof(bx_int16));
      if (data==NULL)
      {
        BX_ERRETURN(BX_E_HOST_MEM_FULL);
      }

      /* Fill data[] with needed properties */
      i=0;
      for (row=FirstRow;row<=LastRow;row++)
      {
        for (col=FirstCol;col<=LastCol;col++)
        {
          data[i++]=DB_RT.Behavior.Mem[row][col];
        }
      }

      assert(i==NumWords);

      /* Write to target behavior memory */
      BX_TRY(BestXDirectRegBlockWrite(handle,
                                      BX_REG_RTBEHAV_DATA_REG,
                      sizeof(bx_int16),
                                      1, /* 1=autoincrement */
                                      (bx_int8ptr) data,
                                      sizeof(bx_int16),
                                      NumWords*sizeof(bx_int16)
                                      ));

      BestXMemFree((void**) &data);

      /* Mark host memory to be identical with HW */
      SetUpdateInfo(BEH_UPD,0,0,0);

      /* Switch to Run-Mode (PCI clock) */
      BX_TRY_PROGRESS(BX_E_OK);
      BX_TRY_PROGRESS(BestXExerciserRunMode(handle,0 /* no soft force */));
    }
  }

  BX_TRY_CATCH
  {
    BX_TRY_PASSED
    {
      /* Switch to progmode successful */
      BX_TRY_PASSED
      {
        /* all functions in between successful */
        BX_TRY_PASSED
        {
          /* switch to runmode successful */
        }
        BX_TRY_FAILED
        {
          /* switch to runmode not successful */
          (void) BestXExerciserRunMode(handle, 1 /* soft force */);
        }
      }
      BX_TRY_FAILED
      {
        /* some function in between not successful */
        if (data)
        {
          BestXMemFree((void**) &data);
        }
        (void) BestXExerciserRunMode(handle, 0 /* no soft force */);
      }
    }
    BX_TRY_FAILED
    {
      /* switch to progmode not successful */
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXRTBehRead
 *
 * Purpose: Reads behavior memory from card to host
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXRTBehRead(bx_handletype  handle,bx_int32 option)
{
  /* Updates behavior memory on host DB */

  BX_DECLARE_FUNCNAME("BestXRTBehRead [rtbehread]");

  BX_TRY_VARS;

  bx_int32 NumLines=0;   /* number of lines to read */

  BX_TRY_BEGIN
  {
    /* Switch to Prog-Mode (DBI clock) */
    BX_TRY_PROGRESS(BestXExerciserProgMode(handle,0 /* no soft force */));

    /* Min and Max registers determine the Column (property) -range 
       to read from
    */
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_RTBEHAV_MIN_REG,sizeof(bx_int16),0));
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_RTBEHAV_MAX_REG,sizeof(bx_int16),BX_RTBEH_MEMWIDTH-1));

    /* Set starting row (i.e. offset) from which to start reading */
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_RTBEHAV_CTR,sizeof(bx_int16),0));

    /* Determine number of lines to read */
    switch (option)
    {
      case BX_EXEREAD_CARD:
        /* 
           The number of lines to read is determined 
           by current HW values.
        */    
        BX_TRY(BestXDirectRegRead(handle, BX_REG_RTBEHAV_CTR0_LOOP_REG,sizeof(bx_int16),&NumLines));
        NumLines++; /* HW value is off by one */
        break;
      case BX_EXEREAD_ALL:
        /* Read complete memory */
        NumLines=BX_RTBEH_MEMDEPTH;
        break;
      case BX_EXEREAD_NONE:
        /* Read nothing */
        NumLines=0;
        break;
      case BX_EXEREAD_USER:
        /* 
           User can determine how many lines are read from card 
           by setting corresponding generic property before.
           Remark:
           During BestXExerciserRead(), the ..GenRead() 
           function must be called after this function,
           because it would overwrites the property with HW-values !!!
         */
        BX_TRY(BestXRTGenGet(handle, BX_RTGEN_NUMBEH, &NumLines));
        break;
      default:
        BX_E_ERROR_MSG_SET("BestXExerciserRead: Invalid value for parameter option");
        BX_TRY_ERROR(BX_E_ERROR);
        break;
    }



    /* Read complete behavior memory from card */
    BX_TRY(BestXDirectRegBlockRead(handle,
                                    BX_REG_RTBEHAV_DATA_REG,
                    sizeof(bx_int16),
                                    1, /* 1=autoincrement */
                                    (bx_int8ptr)DB_RT.Behavior.Mem,
                                    sizeof(bx_int16),
                                    NumLines*BX_RTBEH_MEMWIDTH*sizeof(bx_int16)
                                    ));

    /* Switch to Run-Mode (PCI clock) */
    BX_TRY_PROGRESS(BX_E_OK);
    BX_TRY_PROGRESS(BestXExerciserRunMode(handle,0 /* no soft force */));
  }

  BX_TRY_CATCH
  {
    BX_TRY_PASSED
    {
      /* Switch to progmode successful */
      BX_TRY_PASSED
      {
        /* all functions in between successful */
        BX_TRY_PASSED
        {
          /* switch to runmode successful */
        }
        BX_TRY_FAILED
        {
          /* switch to runmode not successful */
          (void) BestXExerciserRunMode(handle, 1 /* soft force */);
        }
      }
      BX_TRY_FAILED
      {
        /* some function in between not successful */
        (void) BestXExerciserRunMode(handle, 0 /* no soft force */);
      }
    }
    BX_TRY_FAILED
    {
      /* switch to progmode not successful */
    }
  }


  /* chris: 
     Do not change the update info here, because BestXOpen()
     marks the whole memory valid (except first line).
     SetUpdateInfo(BEH_UPD,0,0,0);
  */

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXRTBehCopyLine
 *
 * Purpose: Copies one behavior line to another.
 *          Used only by application.
 *          Do not document.
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXRTBehCopyLine(
  bx_handletype  handle,
  bx_int32 srcline,
  bx_int32 destline
)
{
  BX_DECLARE_FUNCNAME("BestXRTBehCopyLine [rtbehcopyline]");
  
  bx_int32 col;
 
  BX_TRY_VARS;

  BX_TRY_BEGIN
  {
    assert(srcline<BX_RTBEH_MEMDEPTH);    
    assert(destline<BX_RTBEH_MEMDEPTH);    

    /* Set the destination line to default, i.e.
       touch it, so that it will be programmed to card
       by ExerciserProg().
    */
    BX_TRY(BestXRTBehDefaultSet(handle,destline));

    /* Copy line directly in DB */
    for (col=0;col<BX_RTBEH_MEMWIDTH;col++)
    {
      BEHAVIOR(destline)[col]=BEHAVIOR(srcline)[col];
    }
  }
 
  BX_ERRETURN(BX_TRY_RET);
}
